%%%============================================================
%%% Function binsacc() -- Microsaccade Toolbox 0.9
%%% (R-language Version)
%%% Authors: Ralf Engbert, Petra Sinn, Konstantin Mergenthaler, 
%%% and Hans Trukenbrod
%%% Date: February 20th, 2014
%%%
%%% Converted to Matlab by Anna-Katharina Hauperich, 20th June 2019
%%%============================================================
%%%  INPUT:
%%%  msl$table		monocular saccades left eye
%%%  msr$table		monocular saccades right eye  
%%%
%%%  OUTPUT:
%%%  sac$N(1:3)		      number of microsaccades (bin, monol, monor)
%%%  sac$bin(:,1:14)    binocular microsaccades (right eye/left eye)
%%%  sac$monol(:,1:7)   monocular microsaccades of the left eye
%%%  sac$monor(:,1:7)   monocular microsaccades of the right eye
%%%  Basic saccade parameters: (1) onset, (2) end, (3) peak velocity, 
%%%  (4) horizontal component, (5) vertical component, (6) horizontal 
%%%  amplitude, (7) vertical amplitude
%%%---------------------------------------------------------------------
function [sac] = binsaccMSR(sacl,sacr)
  numr = length(sacr(:,1));
  numl = length(sacl(:,1));
  NB = 0;
  NR = 0;
  NL = 0;
  if ( numr*numl>0 )  
    % Determine saccade clusters
    TR = max(sacr(:,2));
    TL = max(sacl(:,2));
    TB = max([TL,TR]);
    s = repmat(0,1,(TB+1));
    SizeSacl = size(sacl);
    for  i= 1:SizeSacl(1)
      left = sacl(i,1):sacl(i,2);
      s(left) = 1;
%       clear left
    end
    SizeSacr = size(sacr);
    for  i = 1:SizeSacr(1)
      right = sacr(i,1):sacr(i,2);
      s(right) = 1;
    end
    s(1) = 0;
    s(TB+1) = 0;
    
    % Find onsets and offsets of microsaccades
    onoff = find(diff(s)~=0);
    ncol=2;
    nrow=length(onoff)/ncol;
    m = reshape(onoff,[ncol,nrow])';
    Sizem = size(m);
    N = Sizem(1);
    
    % Determine binocular saccades
    bin = [];
    monol = [];
    monor = [];
    for i = 1:N
      left  = find( (m(i,1)<=sacl(:,1)) & (sacl(:,2)<=m(i,2)) );
      right = find( (m(i,1)<=sacr(:,1)) & (sacr(:,2)<=m(i,2)) );
      % Binocular saccades
      if ( length(left)*length(right)>0 ) 
        ampr = sqrt( (sacr(right,6)).^2 + (sacr(right,7)).^2 );
        ampl = sqrt( (sacl(left,6)).^2  + (sacl(left,7)).^2 );
        % Determine largest event in each eye
        [~,ir] = max(ampr);
        [~,il] = max(ampl);
        NB = NB + 1;
        bin = [bin;[sacl(left(il),:),sacr(right(ir),:)]];
      else
        % Determine monocular saccades
        if ( length(left)==0 )
          NR = NR + 1;
          ampr = sqrt( (sacr(right,6)).^2 + (sacr(right,7)).^2 );
          [~,ir] = max(ampr);
          monor = [monor;sacr(right(ir),:)];
        end    
        if ( length(right)==0 )
          NL = NL + 1;
          ampl = sqrt( (sacl(left,6)).^2  + (sacl(left,7)).^2 );
          [~,il] = max(ampl);
          monol = [monol;sacl(left(il),:)];
        end
      end
    end  
   else 
    % Special case of exclusively monocular saccades
    if ( numr==0 ) 
      bin = [];
      monor = [];
      monol = sacl;
    end
    if ( numl==0 ) 
      bin = [];
      monol = [];
      monor = sacr;
    end
  end
  clear sac
  sac.N =[NB,NL,NR];
  sac.bin = bin;
  sac.monol = monol;
  sac.monor = monor;
