%%% Detects Microsaccades in eye movement data based on binocular correlations
%%% written by Anna-Katharina Hauperich, July 2019
function MS_BC_BinocularCorrelation(WindowSize, VelocityThresh, RemoveEndsFac, RecFreq)
%% adjust windowsize and removeends for different recording frequencies
if RecFreq ~= 1000
    fact = 1000/RecFreq;
    RemoveEndsFac = RemoveEndsFac/fact;
    WindowSize = WindowSize/fact;
end
removeends = floor(WindowSize/RemoveEndsFac);

if mod(WindowSize,2)~=0
    WindowSize = round(WindowSize);
    if mod(WindowSize,2)~=0
        WindowSize = WindowSize-1;
    end
end

if mod(removeends,2)~=0
    removeends = round(removeends);
end
%% Load Data
load('MS_Parameters.mat')

%% correlation of velocity and acceleration
[BinoCorr] = MS_BC_BinoCorr(ParameterData.Left.Velocity,ParameterData.Right.Velocity,WindowSize);
%% detect periods of high velocity and acceleration correlation within time WindowSize
%use either moving mean or absolute threshold value
% square data
BinoCorr.CorVel = BinoCorr.CorVel.^2;

if ~isnan(VelocityThresh(1))
    
    % take VelocityThresh as threshold
    Corr.Abs.VelTag = zeros(size(BinoCorr.CorVel));
    Corr.Abs.VelTag(BinoCorr.CorVel>(VelocityThresh.^2)) = 1;
    VelThresh = VelocityThresh.^2;
else % use median multiplier for threshold
    
    MedianMultiplier=VelocityThresh(2);
    
    % Calculate threshold based on median multiplier
    Corr.Abs.VelTag = zeros(size(BinoCorr.CorVel));
    Corr.Abs.VelTag(BinoCorr.CorVel>(MedianMultiplier.*nanmedian(BinoCorr.CorVel))) = 1;
    VelThresh = MedianMultiplier.*nanmedian(BinoCorr.CorVel);
end

%Make logical vector
Corr.Abs.CorrTag = logical(Corr.Abs.VelTag');

%%
if sum(isnan(BinoCorr.CorVel))>0
    %% reduce length by certain number of samples either side
    Corr.Abs.CorrTag = ReduceLength(Corr.Abs.CorrTag,removeends)';
    
end

%% Save MS vector and other inrformation to file
MSTagBC = Corr.Abs.CorrTag;

save('MS_Tag.mat', 'MSTagBC')
save('MS_CorrInfo.mat', 'Corr', 'BinoCorr', 'VelThresh')
